/**
 * Polyfill for `document.contains()` method. Notice that according to spec
 * `document.contains` is inclusionary.
 * See https://developer.mozilla.org/en-US/docs/Web/API/Node/contains
 * @param {?Node} node
 * @return {boolean}
 * @this {Document}
 */
function documentContainsPolyfill(node) {
    // Per spec, "contains" method is inclusionary
    // i.e. `node.contains(node) == true`. However, we still need to test
    // equality to the document itself.
    // eslint-disable-next-line local/no-invalid-this
    return node == this || this.documentElement.contains(node);
}

/**
 * Polyfills `HTMLDocument.contains` API.
 * @param {!Window} win
 */
export function install(win) {
    // HTMLDocument is undefined in Internet Explorer 10, but it has Document,
    // so we use that as a fallback.
    const documentClass = win.HTMLDocument || win.Document;
    if (documentClass && !documentClass.prototype.contains) {
        win.Object.defineProperty(documentClass.prototype, 'contains', {
            enumerable: false,
            configurable: true,
            writable: true,
            value: documentContainsPolyfill,
        });
    }
}